/*
    This model is used to represent common functionality across the entire scheduling form.
    It is not bound to any particular API resource, and therefore save and fetch should never be
    called against it.

    The primary purpose of this model is to provide shortcut function for looking up data and
    for views to bind change events against it.
 */
define([
    'underscore',
    'App',
    'User',
    'models/abstract-model',
    'modules/new-appointment-request/resources/constants',
], function(_, app, user, Model, constants) {
    'use strict';


    var ABSTRACT_MESSAGE = 'This function is intended to be overridden by extending model';
    var REQUEST = 'clerk';
    var DIRECT = 'direct';
    var DEFAULT_TIMEZONE = 'America/New_York';


    return Model.extend({

        /**
         * This will not function from this model, but still handles common functionality
         * for extending models.
         * @return {xhr}
         * @throws Error
         */
        save: function() {
            var url = this.getUrl();
            var data = this.createSaveData();
            var options = _.extend({}, this._requestDefaults, {
                url: url,
                wait: true,
            });

            return Model.prototype.save.call(this, data, options);
        },
        /**
         * Flag for when Telehealth features are enabled
         * @return {boolean}
         */
        isTelehealthFeaturesEnabled: function() {
            return _.isUndefined(app.disabledFeatures.get('TH_FEATURE_SET'));
        },

        /**
         * Abstract: Designed to be overridden by extending model
         * @return {string}
         * @throws Error
         */
        getUrl: function() {
            throw Error(ABSTRACT_MESSAGE);
        },

        /**
         * Abstract: Designed to be overridden by extending model
         * @return {*}
         * @throws Error
         */
        createSaveData: function() {
            throw Error(ABSTRACT_MESSAGE);
        },

        /**
         * @return {Backbone.Model|undefined}
         */
        typeOfCare: function() {
            return this.get('typeOfCare');
        },

        /**
         * @return {Backbone.Model|undefined}
         */
        facility: function() {
            return this.get('facility');
        },

        /**
         * @return {Backbone.Model|undefined}
         */
        method: function() {
            return this.get('scheduling-method');
        },

        /**
         * @return {Backbone.Model|undefined}
         */
        clinic: function() {
            return this.get('clinic');
        },

        /**
         * @return {Backbone.Model|undefined}
         */
        getEmailPreferences: function() {
            return this.get('emailPreferences');
        },

        /**
         * Added so extending models do not have to import the user directly
         * @return {Backbone.Model} The user information
         */
        user: function() {
            return user;
        },

        /**
         * @return {Backbone.Collection|undefined}
         */
        pacTeam: function() {
            return this.get('pacTeam');
        },

        /* ====================== *
         |  TYPE OF CARE WRAPPERS |
         * ====================== */

        getCareName: function() {
            var typeOfCare = this.typeOfCare();
            if (typeOfCare) {
                return typeOfCare.get('name');
            }
            return '';
        },

        getCareId: function() {
            var typeOfCare = this.typeOfCare();
            if (typeOfCare) {
                return typeOfCare.get('id');
            }
            return '';
        },

        /**
         * @return {string} Empty if the facility model is not set
         */
        getFacilityName: function() {
            var facility = this.facility();
            if (facility) {
                return facility.get('name');
            }
            return '';
        },

        /**
         * @return {string} Empty if the facility model is not set
         */
        getInstitutionCode: function() {
            var facility = this.facility();
            if (facility) {
                return facility.get('institutionCode');
            }
            return '';
        },

        /**
         * @return {string} Empty if the facility model is not set
         */
        getInstitutionTimezone: function() {
            var facility = this.facility();
            if (facility) {
                return facility.get('institutionTimezone');
            }
            return DEFAULT_TIMEZONE;
        },

        /**
         * @return {string} Empty if the facility model is not set
         */
        getSiteCode: function() {
            var facility = this.facility();
            if (facility) {
                return facility.get('rootStationCode');
            }
            return '';
        },

        getClinicId: function() {
            var clinic = this.clinic();
            if (clinic) {
                return clinic.get('clinicId');
            }
            return '';
        },

        getClinicSecondaryStopCode: function() {
            var clinic = this.clinic();
            if (clinic) {
                return clinic.get('secondaryStopCode');
            }
            return '';
        },

        getClinicName: function() {
            var clinic = this.clinic();
            if (clinic) {
                return clinic.getClinicName() || clinic.getFriendlyLocationName() || '';
            }
            return '';
        },

        /**
         * @return {string} The user id in format specified by getUserType
         */
        getUserId: function() {
            return user.get('id');
        },

        /**
         * @return {{emailAllowed: boolean, email: string}|{}}
         */
        getEmail: function() {
            var emailPreferences = this.get('emailPreferences');
            if (emailPreferences && emailPreferences.get('emailAllowed')) {
                return {
                    emailAllowed: true,
                    email: emailPreferences.get('emailAddress'),
                };
            }

            return {};
        },

        /**
         * Short cut function to get the user id type
         * @return {string}
         */
        getUserIdType: function() {
            return user.get('idType');
        },

        getPrimaryProviderList: function() {
            var pacTeam = this.get('pacTeam');
            if (pacTeam) {
                return pacTeam.getPrimaryProviderList();
            }
            return '';
        },

        /**
         * @return {boolean}
         */
        isExpressCare: function() {
            return this.getCareId() === constants.EXPRESS_CARE_ID;
        },

        /**
         * @return {boolean}
         */
        isPrimaryCare: function() {
            return this.getCareId() === constants.PRIMARY_CARE_ID;
        },

        /**
         * @return {boolean}
         */
        isRequest: function() {
            return this.method() === REQUEST;
        },

        /**
         * @return {boolean}
         */
        isDirect: function() {
            return this.method() === DIRECT;
        },

        isVideoVisit: function() {
            return this.getClinicSecondaryStopCode() === constants.VIDEO_STOP_CODE;
        },
    });
});
